#include <fcntl.h>
#include "httpd.h"
#include "http_config.h"
#include "apr_file_io.h"
#include "apr_file_info.h"
#include "util_filter.h"
#include <stdio.h>
#include <clamav.h>

//#define DEBUG  /*you can turn on this flag if you want to see some debug info in /usr/local/apache2/logs/error_log*/
#define BUCKET_GROUP_SIZE 3   /*in streaming av mode, every 3 buckets in brigade are grouped together to be processed*/
#define INIT_BUFFSIZE_FOR_BOUNDARY 125840000   /*init size of buffer in streamav_ctx*/
#define END_OF_BRIGADE 10     
#define EOS_FOUND 4
#define EOS_NOT_FOUND 5
#define BRIGADE_NOT_EMPTY 11
#define STREAMAV_TYPENO 200
#define RECORD_TIME /*similar to macro DEBUG*/

module AP_MODULE_DECLARE_DATA streamav_module;

/*this is for recognization of content type of current filter.
 *currently the major reason is to correctly deal with zip file or executable file.
 *general text file will be directly scanned with general_scan() function and make the scan process real stream.
 *As for special types as below, current approach is to collect all buckets and create a tmp file, then scan the tmp
 *file with special_scan() function. Of course this is not what we mean by "stream"
 */
typedef enum {
  STREAMAV_GENERIC_TYPE = STREAMAV_TYPENO,
  STREAMAV_TYPE_GZIP,
  STREAMAV_TYPE_ZIP,
  STREAMAV_TYPE_RAR,
  STREAMAV_TYPE_TAR,
  STREAMAV_TYPE_BZ,
  STREAMAV_TYPE_OLE2
} streamav_content_t;


/*context structure, for saving context during streaming process. 
 *I guess all proxy that supports streaming mechanism, can provide context saving approach. 
 *Hence, if streamav will be deployed on other proxy, rather than apache, the context structure, or even structure of
 *the whole program can be reused  I hope so:)
 */
typedef struct streamav_ctx {

  char *buffer;         /*buffer to store data to be scanned.*/
  int curr_buff_size;   /*size of the buffer*/
  int offset;           /*offset to the beginning of the buffer, where now buffer content starts from.*/
  int curr_buff_len;    /*length of content currently stored in buffer*/
  
  int* partcnt;         /*clamav context variable. please check src of clamav for detail*/
  unsigned long int *partoff;
  unsigned long int scanned;

  FILE *tmpFile;        /*tmpFile will be created if we need to scan the complete file, instead of streaming buffer*/

  /*bucket brigade to be passed to next filter, content of the brigade will be created during scanning process*/
  apr_bucket_brigade *bb;

  time_t start;
  time_t end;

} streamav_ctx;


/*localtype structure.  If you would like to, you can write your own daemon code.
 */
typedef struct streamav_local_t {
  struct cl_node *root;
  time_t lastreload;      /*last time to load virus database*/
} streamav_local_t;


typedef struct streamav_config_rec {
  streamav_local_t *local;
  int reload_interval;
  apr_pool_t *pool;
} streamav_config_rec;


/*each filter stores the MIME type of the content it is processing. 
  streamav_magic_s structure is used to recognize the content_type
  variable magic is the specific string of each MIME type descriptor
  variable type is the corresponding number of a certain type
 */
struct streamav_magic_s{
  const char *magic;
  streamav_content_t type;
};


/* you can extense set of MIME type that could be recognized by streamav
 */
static const struct streamav_magic_s streamav_magic[]={
  {"gzip",STREAMAV_TYPE_GZIP},
  {"zip",STREAMAV_TYPE_ZIP},
  {"rar",STREAMAV_TYPE_RAR},
  /*
    {"gif",STREAMAV_TYPE_GIF},
    {"jpeg",STREAMAV_TYPE_JPEG},
  */
  {"bz",STREAMAV_TYPE_BZ},
  {"tar",STREAMAV_TYPE_TAR},
  {"ole",STREAMAV_TYPE_OLE2},
  {NULL,STREAMAV_GENERIC_TYPE}
};


/*initialize context when a new request invoke the streamav filter
  1.init cl_node if clamav engine has not been ready
  2.allocate space for buffer
  3.init bb for next filter
  if fail, return -1; else return 0;
*/
static int streamav_init(ap_filter_t *f, streamav_config_rec *rec)
{
  streamav_ctx *ctx = (streamav_ctx *)f->ctx;
  time_t now;
  
  if(rec == NULL){
    return -1;
  }
  /*no need to do initialization here*/
  if(ctx!=NULL){
    return 0;
  }

  /* allocate a new context structure*/
  f->ctx = apr_pcalloc(f->r->pool,sizeof(streamav_ctx));
  ctx = (streamav_ctx *)f->ctx;

  time(&now);

  /*if clamav engine has not been ready, we do following part*/
  if(rec->local == NULL){

    rec->local = (streamav_local_t *)apr_palloc(rec->pool,sizeof(streamav_local_t));
    rec->local->root = NULL;
    rec->local->lastreload = now;

    /*initialize clamav engine i.e. cl_node root*/
    if(cl_loaddbdir(cl_retdbdir(), &rec->local->root, NULL)!=0){
      fprintf(stderr,"cannot initialize clamav engine\n");
      fflush(stderr);
      return -1;
    }
    cl_buildtrie(rec->local->root);
  
  }

  /*now we don't consider database reload
   *because different version of database of clamav might be slightly different,
   *i don't know whether the interfact to load db might be different too, I just
   *didn't investigate on that. 
   */
  /*  else if ((now - rec->local->lastreload)>rec->reload_interval){
    cl_freetrie(rec->local->root);
    rec->local->root = NULL;

    if(cl_loaddbdir(cl_retdbdir(), &rec->local->root, NULL)!=0){
      fprintf(stderr,"cannot initialize clamav engine\n");
      fflush(stderr);
      return -1;
    }
    cl_buildtrie(rec->local->root);
    
    rec->local->lastreload = now;
  }*/


  /*allocate buffer. Here buffer is used to store both fresh data bucket and boundary data 
    in order to avoid boundry effect*/
  if(!(ctx->buffer=(char *)cli_calloc(INIT_BUFFSIZE_FOR_BOUNDARY, sizeof(char)))){
    fprintf(stderr,"cannot allocate space for ctx->buffer\n");
    fflush(stderr);
    return -1;
  }

  ctx->curr_buff_size = INIT_BUFFSIZE_FOR_BOUNDARY;
  ctx->offset = 0;
  ctx ->curr_buff_len = 0;

  ctx->partcnt = (int *)cli_calloc(rec->local->root->ac_partsigs+1,sizeof(int));
  ctx->partoff = (unsigned long int *)cli_calloc(rec->local->root->ac_partsigs+1,sizeof(unsigned long int));
  ctx->scanned = 0;

  ctx->tmpFile = NULL;

  /*create a new bucket brigade*/
  ctx->bb = apr_brigade_create(f->r->pool, f->c->bucket_alloc);

  return 0;
}

/*Say that we don't need to make special operation on current data, we can scan the data with streaming mechanism
 *scanbuff: buffer storing data to be scanned
 *buffsize: length of content in buffer
 *virname: if data wasnot clean, virname will be set the value of virus name
 *rec: info about clamav engine
 *if buffer is infected, return value is CL_VIRUS, otherwise, CL_CLEAN
 */
static apr_status_t streamav_general_scan(ap_filter_t *f, const char *scanbuff, unsigned int buffsize, const char **virname, streamav_config_rec *rec)
{
  streamav_ctx *ctx = (streamav_ctx *)f->ctx;
  int ret;

  if((ret=cli_bm_scanbuff(scanbuff,buffsize,virname,rec->local->root,ctx->scanned,0,-1))!= CL_VIRUS){
    ret = cli_ac_scanbuff(scanbuff,buffsize,virname,rec->local->root,ctx->partcnt,0,ctx->scanned,ctx->partoff,0,-1);
  }
  
  return ret;
}


/*Say that we need make some special deal with current data, we need to save bucket content into a tmp file and scan the file
  f: filter, context of this filter contains info about tmpFile
  virname:if data was not clean,virname will be set the value of virus name
  rec:info about clamav engine
  type:type of current tmp file
 */
static apr_status_t streamav_special_scan(ap_filter_t *f, const char **virname, streamav_config_rec *rec, streamav_content_t type)
{
  streamav_ctx *ctx = (streamav_ctx *)f->ctx;
  int fd = fileno(ctx->tmpFile);
  int len;

  return cl_scandesc(fd,virname,(unsigned long int *)&len,rec->local->root,NULL,CL_ARCHIVE);
  //return CL_CLEAN;
}


/*remove all buckets in f->ctx->bb*/
static apr_status_t streamav_cleanbrigade(ap_filter_t *f)
{
  apr_bucket *curr_bucket;
  streamav_ctx *ctx = (streamav_ctx *)f->ctx;

#ifdef DEBUG
  fprintf(stderr,"\n now in cleanbrigade\n");
  fflush(stderr);
#endif
  
  APR_BRIGADE_FOREACH(curr_bucket, ctx->bb){
    
#ifdef DEBUG
    fprintf(stderr,"some buckets to be removed");
    fflush(stderr);
#endif

    apr_bucket_delete(curr_bucket);
  }
  
  return APR_SUCCESS;
}


/*cleanup everything. 
 *This function is usually called at the end of program:1.virus found 2.data clean,and we reach end of request
 *only difference of cleanup action under CL_VIRUS and CL_CLEAN is that
 *we need to insert a EOS bucket in ctx->bb,if virus is found
 */
static apr_status_t streamav_cleanup(ap_filter_t *f,apr_bucket_brigade *bb,int status)
{
  streamav_ctx *ctx = (streamav_ctx *)f->ctx;
  apr_bucket *curr_bucket;


  if(status == CL_VIRUS){
    fprintf(stderr,"\nthis is infected req\n");
    fflush(stderr);
    //streamav_cleanbrigade(f);
    APR_BRIGADE_INSERT_TAIL(ctx->bb,apr_bucket_eos_create(f->c->bucket_alloc));
    ap_pass_brigade(f->next,ctx->bb);
    apr_brigade_cleanup(ctx->bb);
  }
  
  free(ctx->buffer);
  free(ctx->partcnt);
  free(ctx->partoff);
  
  apr_brigade_cleanup(bb);
  return APR_SUCCESS;
}


/*function used to handle boundary effect. 
 * boundary effect: it is possible that a virus signature was split into two buckets. under this situation, we 
 *need to handle boundary effect in order not to miss the signature. i.e. we need to save enough data at this time
 *and integrate the data with the content we got in next time
 */
static int streamav_boundary_handler(ap_filter_t *f,streamav_config_rec *rec)
{
  streamav_ctx *ctx = f->ctx;
  int bytes;
  
  /*if length of current buffer is smaller than root->matpatlen, we need to save all buffer
   *otherwise, we just need to save content of size root->maxpatlen, because this is enough for boundary 
   *handling
   */
  if(rec->local->root->maxpatlen>ctx->curr_buff_len)
    bytes = ctx->curr_buff_len;
  else bytes = rec->local->root->maxpatlen;
  
  //  (ctx->root->maxpatlen>buffsize) ? bytes : buffsize,ctx->root->maxpatlen;

#ifdef DEBUG
  fprintf(stderr,"\nthis is in boundary, and boundary bytes is %d\n",bytes);
  fflush(stderr);
#endif

  /*update the content's starting position in the buffer*/
  ctx->offset += ctx->curr_buff_len - bytes;

  /*update the bytes we have scanned p.s. content saved for boundary handling cannot be taken as
   *content that has been scanned, because it will be used again next time
   */
  ctx->scanned += ctx->curr_buff_len - bytes;
  
#ifdef DEBUG
  fprintf(stderr,"\n now offset is %d, scanned is %d\n",ctx->offset,ctx->scanned);
  fflush(stderr);
#endif

  ctx->curr_buff_len = bytes;
  
  return bytes;
  
}


/*
 */
static streamav_config_rec *streamav_get_module_config(request_rec *r)
{
  streamav_config_rec *result;
  if(r == NULL){
    fprintf(stderr,"this is error in get module config\n");
    fflush(stderr);
    return NULL;
  }
  
  result = (streamav_config_rec *)ap_get_module_config(r->per_dir_config,&streamav_module);
  if (result != NULL){
    return result;
  }
  
  fprintf(stderr,"this is error in get module config\n");
  fflush(stderr);
  return NULL;

}


/*write content of each bucket into a tmpfile until we reach EOS. 
 */
int streamav_save_brigade(ap_filter_t *f, apr_bucket_brigade *bb)
{
  streamav_ctx *ctx = (streamav_ctx *)f->ctx;
  const char *buffer;
  apr_size_t len;
  apr_bucket *curr_bucket, *tmp_bucket;

  if(ctx->tmpFile==NULL){
    ctx->tmpFile = tmpfile();
  }

  fseek(ctx->tmpFile,0,SEEK_END);

  APR_BRIGADE_FOREACH(curr_bucket,bb){
    if((!APR_BUCKET_IS_EOS(curr_bucket))&&(!APR_BUCKET_IS_FLUSH(curr_bucket))){
      apr_bucket_read(curr_bucket,&buffer,&len,APR_NONBLOCK_READ);
      
      if(fwrite(buffer,sizeof(char),len,ctx->tmpFile)!=len){
	fprintf(stderr,"\nerror writing file\n");
	fflush(stderr);
      }
     
      /*insert current bucket into ctx->bb, pending to be passed to next filter
       *if we found virus in later scanning process, we will remove all buckets in ctx->bb
       */
      apr_bucket_copy(curr_bucket,&tmp_bucket);
      APR_BRIGADE_INSERT_TAIL(ctx->bb,tmp_bucket);

      /*delete curret bucket from bb*/
      apr_bucket_delete(curr_bucket);
    }

    else if(APR_BUCKET_IS_FLUSH(curr_bucket)){
      apr_bucket_delete(curr_bucket);
    }
    
    else {
#ifdef DEBUG
      fprintf(stderr,"\nnow in streamav_save_brigade(),EOS was found\n");
      fflush(stderr);
#endif
      //      fclose(ctx->tmpFile);
      return EOS_FOUND;
    }
  }

  return EOS_NOT_FOUND;
}


/*read up to 3 buckets from bb and write the content into buffer
 *also, we need to insert these 3 visited bucket into ctx->bb, pending to be passed to
 *next filter
 */
static int streamav_save_group_bucket(ap_filter_t *f, apr_bucket_brigade *bb)
{
  const char *buffer;            /*buffer to store data content of current apache bucket*/
  apr_size_t len;                /*length of buffer*/
  streamav_ctx *ctx = (streamav_ctx *)f->ctx;
  apr_bucket *curr_bucket,*tmp_bucket;
  
  int i;
  
  for(i=0;i<BUCKET_GROUP_SIZE;i++){

    curr_bucket = APR_BRIGADE_FIRST(bb);
    if(curr_bucket==APR_BRIGADE_SENTINEL(bb)){
      return END_OF_BRIGADE;
    }
    
    else{

      /*we just handle data buckets and group them*/
      if((!APR_BUCKET_IS_EOS(curr_bucket))&&(!APR_BUCKET_IS_FLUSH(curr_bucket))){
      
	apr_bucket_read(curr_bucket,&buffer,&len,APR_NONBLOCK_READ); 

#ifdef DEBUG
	fprintf(stderr,"\nthis is a bucket, and length is %d\n",len);
	fflush(stderr);
#endif
	
	/************************************************************************************/
			  
	if((ctx->curr_buff_len + len)>ctx->curr_buff_size){
	  ctx->buffer = (char *)realloc(ctx->buffer, (ctx->curr_buff_len+len)*sizeof(char));
	  ctx->curr_buff_size = ctx->curr_buff_len + len;
	  memcpy((char *)ctx->buffer,(char *)ctx->buffer+ctx->offset,ctx->curr_buff_len);
	  ctx->offset = 0;
	  memcpy((char *)ctx->buffer+ctx->curr_buff_len,buffer,len);
	}
	
	/* if there is no enough space after current buffer content, we need to remove current content to the very
	 * beginning of the ctx->buffer
	 */
	else if((ctx->offset+len) > ctx->curr_buff_size){
	  
#ifdef DEBUG
	  fprintf(stderr,"\nnow is a new group bucket, and we need to move the old data to the beginning, offset is %d\n", ctx->offset);
	  fflush(stderr);
#endif
	
	  memcpy((char *)ctx->buffer+ctx->offset,ctx->buffer,ctx->curr_buff_len);
	  ctx->offset = 0;
	  memcpy((char *)ctx->buffer+ctx->curr_buff_len,buffer,len);
	}
      
	else{
	  memcpy((char *)ctx->buffer+ctx->offset+ctx->curr_buff_len,buffer,len);
	}
      
	ctx->curr_buff_len += len;
	/***********************************************************************************/
	
#ifdef DEBUG
	fprintf(stderr,"\n now write this buffer into ctx->bb for temp \n");
	fflush(stderr);
#endif
      
	//	ap_fwrite(f->next,ctx->bb,buffer,len);
	apr_bucket_copy(curr_bucket,&tmp_bucket);
	APR_BRIGADE_INSERT_TAIL(ctx->bb,tmp_bucket);

	/*delete curret bucket from bb*/
	apr_bucket_delete(curr_bucket);
      }

      else if(APR_BUCKET_IS_FLUSH(curr_bucket)){
	apr_bucket_delete(curr_bucket);
      }
      
      else {
#ifdef DEBUG
	fprintf(stderr,"\nnow in streamav_save_group_bucket(),EOS was found\n");
	fflush(stderr);
#endif
	return EOS_FOUND;
      }
    }
  }

  return BRIGADE_NOT_EMPTY;
}


/*determine type of content in filter f.
 *return value is streamav_content_t
 */
static int streamav_contenttype(ap_filter_t *f)
{
  int i;
  
  for(i=0;streamav_magic[i].magic;i++){
    if(strstr(f->r->content_type,streamav_magic[i].magic)){
      return streamav_magic[i].type;
    }
  }
  
  return streamav_magic[i].type;
}


/*send ctx->bb to next filter
 */
static void streamav_send(ap_filter_t *f)
{
  streamav_ctx *ctx = (streamav_ctx *)f->ctx;
  
  APR_BRIGADE_INSERT_TAIL(ctx->bb,apr_bucket_flush_create(f->c->bucket_alloc));
  ap_pass_brigade(f->next,ctx->bb);
}


static apr_status_t streamav_filter(ap_filter_t *f,apr_bucket_brigade *bb)
{

  streamav_ctx *ctx= (streamav_ctx *)f->ctx;      /*filter context*/
  apr_bucket *curr_bucket;      
  const char* virname; 

  streamav_config_rec *rec;
  
  int flag;
  int content_type;

  struct timeval now;
  struct timeval current;
  //struct timezone tz;

  rec = streamav_get_module_config(f->r);

  /*initialize filter context if this filter is called for the 1st time. i.e. this is a new request
   *because each request might be splited into several brigades, each of which will invoke the same filter instance. so we just need to 
   *do intialization for only once when first brigade arrives.
   */
  if(ctx==NULL){
    streamav_init(f,rec);
    ctx = (streamav_ctx *)f->ctx;

    time(&ctx->start);
#ifdef RECORD_TIME
    fprintf(stderr,"\n\nthis is a new req %s\ncontent type is %s\nstart time is%d\n",f->r->the_request,f->r->content_type,ctx->start);
    fflush(stderr);
#endif

  }

  /*if this request transports special files like zip file, etc. we need to do av operation with special_scan function
   */
  if(1){
    //if((content_type = streamav_contenttype(f))!=STREAMAV_GENERIC_TYPE){

    gettimeofday(&now,NULL);

    /*collect all buckets in this brigade, until we reach EOS bucket. 
     *and write the content into a tmp file(variable in ctx)
     *and buffer content in ctx->bb. 
     *if content is clean, we could directly pass ctx->bb to next filter
     */
    if(streamav_save_brigade(f,bb)==EOS_FOUND){

      gettimeofday(&current,NULL);
      fprintf(stderr,"\ntime cost of save brigade is%ld, %ld",current.tv_sec-now.tv_sec,(current.tv_usec-now.tv_usec));
      fflush(stderr);

      /*if EOS is reached, that means we have had complete file and can do scan work now*/
      fseek(ctx->tmpFile,0,SEEK_SET);

      gettimeofday(&now,NULL);

      /*if file is infected, we need to cleanup everything and returrn hint information to client*/
      if(streamav_special_scan(f,&virname,rec,content_type)==CL_VIRUS){
	gettimeofday(&current,NULL);
	fprintf(stderr,"\ntime cost of scan file is%ld, %ld",current.tv_sec-now.tv_sec,(current.tv_usec-now.tv_usec));
	fflush(stderr);
	fclose(ctx->tmpFile);
	streamav_cleanbrigade(f);
	ap_fputs(f->next,ctx->bb,"<html><b>StreamAV test result</b><br>This is an infected webpage<br><br>VIRUS FOUND: <b>");
	ap_fwrite(f->next,ctx->bb,virname,strlen(virname));
	ap_fputs(f->next,ctx->bb,"</b></html>");
	ap_pass_brigade(f->next,ctx->bb);
	streamav_cleanup(f,bb,CL_VIRUS);
	
	time(&ctx->end);
#ifdef RECORD_TIME
	fprintf(stderr,"\nthis is a infected file,VIRUS FOUND:%s\nend time:%d\ntime cost is:%d\n",virname,ctx->end,ctx->end-ctx->start);
	fflush(stderr);
#endif

	return APR_SUCCESS;
      }

      /*if file is clean, we can send it to client*/
      else{

	gettimeofday(&current,NULL);
	fprintf(stderr,"\ntime cost of scan file is%ld, %ld",current.tv_sec-now.tv_sec,(current.tv_usec-now.tv_usec));
	fflush(stderr);

	time(&ctx->end);
#ifdef RECORD_TIME
	fprintf(stderr,"\nnow send file to client. reaction time is:%ld\n",ctx->end-ctx->start);
	fflush(stderr);
#endif
	streamav_send(f);

	time(&ctx->end);
#ifdef RECORD_TIME
	fprintf(stderr,"\nnow finish sending. time is:%d\n",ctx->end-ctx->start);
	fflush(stderr);
#endif
	//	streamav_cleanup(f,bb,CL_CLEAN);
      }
    }
    
    else {
      gettimeofday(&current,NULL);
      fprintf(stderr,"\ntime cost of save brigade is%ld, %ld",current.tv_sec-now.tv_sec,(current.tv_usec-now.tv_usec));
      fflush(stderr);
    }
    
  }

  /*if the request transports general file, which means we can do av operation with streaming mode, we can do that with general_scan
    operation
   */

  else{

    do{
      gettimeofday(&now,NULL);
      
      flag = streamav_save_group_bucket(f,bb);

      gettimeofday(&current, NULL);
      fprintf(stderr,"\ntime cost of save group bucket is%ld, %ld",current.tv_sec-now.tv_sec,(current.tv_usec-now.tv_usec));
      fflush(stderr);

#ifdef DEBUG
      fprintf(stderr,"\n flag is %d",flag);
      fflush(stderr);    
#endif

      gettimeofday(&now,NULL);

      if(streamav_general_scan(f,ctx->buffer+ctx->offset,ctx->curr_buff_len,&virname,rec)==CL_VIRUS){

	gettimeofday(&current,NULL);
	fprintf(stderr,"\ntime cost of scan buffer is%ld, %ld",current.tv_sec-now.tv_sec,(current.tv_usec-now.tv_usec));
	fflush(stderr);

#ifdef DEBUG
	fprintf(stderr,"\nthis is a infected req\n");
	fflush(stderr);
#endif
	
	streamav_cleanbrigade(f);
	ap_fputs(f->next,ctx->bb,"<html><b>StreamAV test result</b><br>This is an infected webpage<br><br>VIRUS FOUND: <b>");
	ap_fwrite(f->next,ctx->bb,virname,strlen(virname));
	ap_fputs(f->next,ctx->bb,"</b></html>");
	ap_pass_brigade(f->next,ctx->bb);
	streamav_cleanup(f,bb,CL_VIRUS);
	
	time(&ctx->end);
#ifdef RECORD_TIME
	fprintf(stderr,"\nthis is a infected file,VIRUS FOUND:%s\nend time:%d\ntime cost is:%d\n",virname,ctx->end,ctx->end-ctx->start);
	fflush(stderr);
#endif

	return APR_SUCCESS;
      }

      gettimeofday(&current,NULL);
      fprintf(stderr,"\ntime cost of scan buffer is%ld, %ld",current.tv_sec-now.tv_sec,(current.tv_usec-now.tv_usec));
      fflush(stderr);

#ifdef DEBUG
      fprintf(stderr,"\n now send this bucket brigade to client \n,flag is %d",flag);
      fflush(stderr);    
#endif

      time(&ctx->end);
#ifdef RECORD_TIME
      fprintf(stderr,"\nnow send file to client. reaction time is:%ld\n",ctx->end-ctx->start);
      fflush(stderr);
#endif

      streamav_send(f);

      time(&ctx->end);
#ifdef RECORD_TIME
      fprintf(stderr,"\nnow finish sending. time is:%d\n",ctx->end-ctx->start);
      fflush(stderr);
#endif

      if(flag==EOS_FOUND){
#ifdef DEBUG
	fprintf(stderr,"\nthis is in streamav_filter().  EOS found in general scan\n");
	fflush(stderr);
#endif
	break;
      }

      streamav_boundary_handler(f,rec);
      
      time(&ctx->end);
#ifdef RECORD_TIME
      fprintf(stderr,"\nnow finish boundary. time is:%d\n",ctx->end-ctx->start);
      fflush(stderr);
#endif
      
      if(APR_BRIGADE_EMPTY(bb)){
	break;
      }

    } while(1);
    
  }

  /*now bb has only EOS buckets, we send out EOS(sign of end of stream) to the client, in order to close connection to client*/
  APR_BRIGADE_FOREACH(curr_bucket,bb){
    if(APR_BUCKET_IS_EOS(curr_bucket)){
      APR_BRIGADE_INSERT_TAIL(ctx->bb,apr_bucket_eos_create(f->c->bucket_alloc));
      ap_pass_brigade(f->next,ctx->bb);
      streamav_cleanup(f,ctx->bb,CL_CLEAN);
      
      time(&ctx->end);
#ifdef RECORD_TIME
      fprintf(stderr,"\nthis is a clean file,end time:%d\ntime cost is:%d\n",ctx->end,ctx->end-ctx->start);
      fflush(stderr);
#endif

    }
    else {
      fprintf(stderr,"this is a wired bucket\n");
      fflush(stderr);
    }
  }    

  apr_brigade_cleanup(bb);
  
  return APR_SUCCESS;  

}  
    



static void* streamav_create_dir_config(apr_pool_t *p, char *d)
{
  streamav_config_rec *cfg;
  cfg = (streamav_config_rec *)apr_palloc(p,sizeof(streamav_config_rec));
  cfg->local = NULL;
  cfg->reload_interval = 0;
  cfg->pool = p;
  
  return cfg;
}

static void* streamav_merge_dir_config(apr_pool_t *p, void *pp, void *cp)
{
  streamav_config_rec *parent = (streamav_config_rec *)pp;
  streamav_config_rec *child = (streamav_config_rec *)cp;
  
  child->local = parent->local;
  child->reload_interval = parent->reload_interval;
	
  return child;
}

static void streamav_register_hooks(apr_pool_t *p)
{
  //    ap_hook_handler(streamav_handler, NULL, NULL, APR_HOOK_MIDDLE);
    ap_register_output_filter("STREAMAV", streamav_filter, NULL,
	AP_FTYPE_CONTENT_SET);
} /* clamav_register_hooks */

module AP_MODULE_DECLARE_DATA streamav_module = {
    STANDARD20_MODULE_STUFF, 
    streamav_create_dir_config,
                           /* create per-dir    config structures */
    streamav_merge_dir_config,
                           /* merge  per-dir    config structures */
    NULL,                  /* create per-server config structures */
    NULL,                  /* merge  per-server config structures */
    NULL,           /* table of config file commands       */
    streamav_register_hooks  /* register hooks                      */
};
